import subprocess
import sys
import winreg
import json
from pathlib import Path

try:
    from rich import pretty, traceback
except ImportError:
    subprocess.check_call([sys.executable, "-m", "pip", "install", "rich"])
finally:
    from rich import pretty, traceback

try:
    import win32api
except ImportError:
    subprocess.check_call([sys.executable, "-m", "pip", "install", "pywin32"])
    print()
    print("Reopen the script!")
    quit(0)

try:
    import pycryptodome  # noqa: F401, cryptography.py needs this
except ImportError:
    subprocess.check_call([sys.executable, "-m", "pip", "install", "pycryptodome"])

from cryptography import is_valid_key
from img3 import IMG3Archive

# doubles as allowlist, i.e. filetypes other than these would be ignored
filetypes = {"wdr": "odr", "wft": "oft"}

pretty.install()
traceback.install()


def get_reg_value(root, path, name):
    try:
        with winreg.OpenKey(root, path) as key:
            return winreg.QueryValueEx(key, name)[0]
    except FileNotFoundError:
        return None

def take_inputs():
    game_folder = get_reg_value(
        winreg.HKEY_LOCAL_MACHINE,
        r"SOFTWARE\WOW6432Node\Rockstar Games\Grand Theft Auto IV",
        "InstallFolder",
    )

    if game_folder is None:
        print("Could not automatically detect game folder")
        game_folder = Path(input("Game Folder: ").strip('"'))
    else:
        print("Detected Game Folder: ", game_folder)
        while True:
            confirm = input("Continue? (yes[Y]/no[n]): ").casefold()
            if confirm in ("n", "no"):
                game_folder = Path(input("Game Folder: ").strip('"'))
                break
            elif confirm in ("y", "yes"):
                game_folder = Path(game_folder)
                break

    if not game_folder.exists():
        print("Game Folder doesn't exists")
        quit(1)
    maps_folder = game_folder / "pc/data/maps"
    if not maps_folder.exists():
        print("pc/data/maps folder doesn't exists")
        quit(1)

    wpl_folder = Path(input("WPL Folder: ").strip('"'))
    if not wpl_folder.exists():
        print("WPL Folder doesn't exists")
        quit(1)
    asset_folder = Path(input("Asset Folder: ").strip('"'))
    if not asset_folder.exists():
        print("Asset Folder doesn't exists")
        quit(1)

    return game_folder, maps_folder, wpl_folder, asset_folder


def get_objs_to_search(wpl: Path) -> set:
    objs = set()
    with wpl.open(encoding="ascii") as reader:
        line = reader.readline()

        while line and not line.startswith("inst"):
            line = reader.readline()
        line = reader.readline()  # inst
        while line:
            if line.startswith("#"):
                continue
            if line.startswith("end"):
                break
            s = line.split(", ")
            objs.add(s[7].casefold())
            line = reader.readline()
    return objs


def sort_models(iter_item: str):
    name, ext = iter_item.rsplit(".", maxsplit=1)
    return ext, name


def search(objs: set[str], map_folder: Path, assets: list[str], aes_key: str | bytes) -> dict[Path, list[str]]:
    missing = {}
    for img_path in map_folder.rglob("*.img"):
        print(f"Searching in {img_path}")
        img = IMG3Archive(img_path, aes_key)
        missing_items = []
        for item in img.table.items:
            fullname = item.name.casefold()
            name, suffix = fullname.rsplit(".", maxsplit=1)
            if suffix not in filetypes:
                continue
            if name in objs:
                of_name = name + "." + filetypes[suffix]
                if of_name not in assets:
                    missing_items.append(item.name)
        if missing_items:
            missing[str(img_path)] = sorted(missing_items, key=sort_models)
    print("Finished searching.")
    return missing


# https://stackoverflow.com/a/68774871
def get_file_version(path):
    info = win32api.GetFileVersionInfo(path, "\\")
    ms = info["FileVersionMS"]
    ls = info["FileVersionLS"]
    return (
        win32api.HIWORD(ms),
        win32api.LOWORD(ms),
        win32api.HIWORD(ls),
        win32api.LOWORD(ls),
    )


aes_key_offset = {
    "1.0.7.0": 0xBE7540,
    "1.0.8.0": 0xC95FD8,
    "1.2.0.59": 0xC5B73C,
}


def get_aes_key(game_exe: Path, key_offset: int) -> bytes:
    with game_exe.open("rb") as reader:
        reader.seek(key_offset)
        return reader.read(32)


def main() -> dict[Path, list[str]]:
    game_folder, maps_folder, wpl_folder, asset_folder = take_inputs()
    game_exe = game_folder / "GTAIV.exe"
    game_version = ".".join(str(elem) for elem in get_file_version(str(game_exe)))
    key_offset = aes_key_offset[game_version]
    key = get_aes_key(game_exe, key_offset)
    if not is_valid_key(key):
        raise ValueError("Invalid Key")

    objs = set()
    for wpl in wpl_folder.glob("*.opl"):
        objs.update(get_objs_to_search(wpl))

    files = [x.name for x in asset_folder.iterdir() if x.is_file()]

    return search(objs, maps_folder, files, aes_key=key)


if __name__ == "__main__":
    missing = main()
    pretty.pprint(missing)
    with open("missing_models.json", "w") as writer:
        json.dump(missing, writer)
